// Popup script - handles UI interactions

document.addEventListener('DOMContentLoaded', async () => {
  await init();
});

async function init() {
  // Show loading state
  const authStatus = document.getElementById('authStatus');
  if (authStatus) {
    authStatus.innerHTML = '<div class="loading">Checking authentication...</div>';
  }

  // Check auth status
  await checkAuthStatus();

  // Setup event listeners
  setupEventListeners();

  // Load user stats (only if authenticated)
  const mainContent = document.getElementById('mainContent');
  if (mainContent && mainContent.style.display !== 'none') {
    await loadUserStats();
  }
}

// Check authentication status
async function checkAuthStatus() {
  try {
    // Method 1: Try to get auth from any open pixymod.com tab
    const tabs = await chrome.tabs.query({ url: 'https://pixymod.com/*' });

    if (tabs.length > 0) {
      try {
        // Inject script to check auth state by looking at DOM
        const results = await chrome.scripting.executeScript({
          target: { tabId: tabs[0].id },
          func: () => {
            // Method 1: Look for user email in the page
            const pageText = document.body.innerText || '';
            const emailRegex = /[\w\.-]+@[\w\.-]+\.\w+/;
            const emailMatch = pageText.match(emailRegex);

            // Method 2: Look for specific user elements
            const userEmailEl = document.querySelector('.userEmail, .user-email, [class*="userEmail"]');
            const userNameEl = document.querySelector('.userName, .user-name, [class*="userName"]');
            const avatarEl = document.querySelector('.navAvatar, .avatar, [class*="avatar"]');
            const logoutBtn = Array.from(document.querySelectorAll('button, a')).find(
              el => el.textContent && (el.textContent.includes('Logout') || el.textContent.includes('Sign Out'))
            );
            const verifiedBadge = Array.from(document.querySelectorAll('span, div')).find(
              el => el.textContent && el.textContent.includes('VERIFIED')
            );

            // Method 3: Check if there's a profile link
            const profileLink = Array.from(document.querySelectorAll('a')).find(
              el => el.href && (el.href.includes('/profile') || el.href.includes('/user'))
            );

            // If we find any of these indicators, user is logged in
            if (emailMatch || userEmailEl || userNameEl || avatarEl || logoutBtn || verifiedBadge || profileLink) {
              let email = 'User';
              let displayName = 'User';

              if (emailMatch) {
                email = emailMatch[0];
                displayName = email.split('@')[0];
              } else if (userEmailEl) {
                email = userEmailEl.textContent.trim();
                displayName = email.split('@')[0];
              } else if (userNameEl) {
                displayName = userNameEl.textContent.trim();
              }

              return {
                isAuthenticated: true,
                user: {
                  email: email,
                  displayName: displayName
                }
              };
            }

            // Check if Sign In button exists (indicates not logged in)
            const signInBtn = Array.from(document.querySelectorAll('button, a')).find(
              el => el.textContent && (el.textContent.includes('Sign In') || el.textContent.includes('Login'))
            );

            if (signInBtn && signInBtn.offsetParent !== null) {
              // Sign in button is visible, user is not logged in
              return { isAuthenticated: false, user: null };
            }

            // Can't determine, assume not logged in
            return { isAuthenticated: false, user: null };
          }
        });

        if (results && results[0] && results[0].result) {
          const authState = results[0].result;
          if (authState.isAuthenticated) {
            
            showAuthenticatedUI(authState.user);
            // Also notify background to store this
            chrome.runtime.sendMessage({
              action: 'storeAuth',
              user: authState.user
            }).catch(() => { });
            return;
          }
        }
      } catch (injectError) {
        
      }
    }

    // Method 2: Check with background script (which checks cookies)
    // Increase timeout to 20 seconds to allow token retrieval
    const response = await Promise.race([
      chrome.runtime.sendMessage({ action: 'checkAuth' }),
      new Promise((_, reject) =>
        setTimeout(() => reject(new Error('Timeout')), 20000)
      )
    ]);

    if (response && response.success !== false && response.isAuthenticated) {
      showAuthenticatedUI(response.user);
    } else {
      showUnauthenticatedUI();
    }
  } catch (error) {
    console.error('Auth check error:', error);
    showUnauthenticatedUI();
  }
}

// Show authenticated UI
function showAuthenticatedUI(user) {
  const authSection = document.getElementById('authSection');
  const mainContent = document.getElementById('mainContent');
  const refreshBtn = document.getElementById('refreshAuthBtn');

  authSection.style.display = 'none';
  mainContent.style.display = 'block';
  if (refreshBtn) {
    refreshBtn.style.display = 'none';
  }

  // Update user info
  if (user) {
    const userNameEl = document.getElementById('userName');
    const userEmailEl = document.getElementById('userEmail');

    if (userNameEl) {
      userNameEl.textContent = user.displayName || user.email?.split('@')[0] || 'User';
    }
    if (userEmailEl) {
      userEmailEl.textContent = user.email || '';
    }

    // User avatar
    const avatar = document.getElementById('userAvatar');
    if (avatar) {
      if (user.photoURL) {
        avatar.innerHTML = `<img src="${user.photoURL}" alt="Avatar">`;
      } else {
        const initialEl = avatar.querySelector('#userInitial');
        if (initialEl) {
          initialEl.textContent = (user.email?.[0] || 'U').toUpperCase();
        }
      }
    }

    // Premium badge
    const badge = document.getElementById('userBadge');
    if (badge) {
      if (user.hasActiveSubscription) {
        badge.innerHTML = '<span class="badge premium">Premium</span>';
        const premiumCta = document.getElementById('premiumCta');
        if (premiumCta) premiumCta.style.display = 'none';
      } else {
        badge.innerHTML = '<span class="badge free">Free</span>';
        const premiumCta = document.getElementById('premiumCta');
        if (premiumCta) premiumCta.style.display = 'block';
      }
    }
  }
}

// Show unauthenticated UI
function showUnauthenticatedUI() {
  const authSection = document.getElementById('authSection');
  const mainContent = document.getElementById('mainContent');

  authSection.style.display = 'block';
  mainContent.style.display = 'none';

  const authStatus = document.getElementById('authStatus');
  authStatus.innerHTML = `
    <div class="auth-prompt">
      <p>Sign in to download premium Freepik assets</p>
      <button class="auth-btn" id="signInBtn">Sign In</button>
      <p style="margin-top: 12px; font-size: 11px; color: #6b7280;">
        Already signed in? Click refresh below
      </p>
    </div>
  `;

  // Show refresh button
  const refreshBtn = document.getElementById('refreshAuthBtn');
  if (refreshBtn) {
    refreshBtn.style.display = 'block';
    refreshBtn.onclick = async () => {
      refreshBtn.textContent = '🔄 Refreshing...';
      refreshBtn.disabled = true;

      // Show loading state
      const authStatus = document.getElementById('authStatus');
      if (authStatus) {
        authStatus.innerHTML = '<div class="loading">Checking authentication...</div>';
      }

      // Try to open pixymod.com if not already open, or refresh existing tab
      const tabs = await chrome.tabs.query({ url: 'https://pixymod.com/*' });
      if (tabs.length === 0) {
        await chrome.tabs.create({ url: 'https://pixymod.com/freepik-downloader' });
        // Wait for page to load
        await new Promise(resolve => setTimeout(resolve, 3000));
      } else {
        // Refresh the existing tab
        await chrome.tabs.reload(tabs[0].id);
        await new Promise(resolve => setTimeout(resolve, 2000));
      }

      await checkAuthStatus();
      refreshBtn.textContent = '🔄 Refresh';
      refreshBtn.disabled = false;
    };
  }

  // Add sign in button listener
  const signInBtn = document.getElementById('signInBtn');
  if (signInBtn) {
    signInBtn.addEventListener('click', handleSignIn);
  }
}

// Handle sign in
async function handleSignIn() {
  // Open freepik-downloader page - login modal will appear automatically
  chrome.tabs.create({ url: 'https://pixymod.com/freepik-downloader' });

  // Wait a bit then refresh auth status
  setTimeout(async () => {
    await checkAuthStatus();
  }, 2000);

  // Don't close immediately, let user see the page
  setTimeout(() => {
    window.close();
  }, 500);
}

// Setup event listeners
function setupEventListeners() {
  // Download current tab button
  const downloadCurrentBtn = document.getElementById('downloadCurrentBtn');
  if (downloadCurrentBtn) {
    downloadCurrentBtn.addEventListener('click', handleDownloadCurrentTab);
  }

  // Open Freepik button
  document.getElementById('openFreepikBtn').addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://www.freepik.com' });
  });

  // View history button
  document.getElementById('viewHistoryBtn').addEventListener('click', async () => {
    const history = await getDownloadHistory();
    showHistoryModal(history);
  });

  // Upgrade button
  const upgradeBtn = document.getElementById('upgradeBtn');
  if (upgradeBtn) {
    upgradeBtn.addEventListener('click', () => {
      chrome.tabs.create({ url: 'https://pixymod.com/pricing' });
    });
  }

  // Settings button
  document.getElementById('settingsBtn').addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://pixymod.com/profile' });
  });

  // Help link
  document.getElementById('helpLink').addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: 'https://pixymod.com/how-to-download' });
  });

  // Check current tab on load
  checkCurrentTab();
}

// Check if current tab is a Freepik asset page
async function checkCurrentTab() {
  try {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tabs.length > 0) {
      const currentTab = tabs[0];
      const url = currentTab.url || '';

      // Check if it's a Freepik asset page
      const assetPatterns = [
        /freepik\.com\/(free|premium)-(photo|vector|psd|ai|video)\//,
        /freepik\.com\/.*\/.*\.html/,
      ];

      const isFreepikAsset = assetPatterns.some(pattern => pattern.test(url));

      const downloadSection = document.getElementById('downloadSection');
      if (isFreepikAsset && downloadSection) {
        downloadSection.style.display = 'block';

        // Show URL info (truncated)
        const urlInfo = document.getElementById('currentUrlInfo');
        if (urlInfo) {
          const title = currentTab.title || 'Freepik Asset';
          const truncatedTitle = title.length > 40 ? title.substring(0, 40) + '...' : title;
          urlInfo.textContent = truncatedTitle;
          urlInfo.style.display = 'block';
        }
      } else if (downloadSection) {
        downloadSection.style.display = 'none';
      }
    }
  } catch (error) {
    console.error('Error checking current tab:', error);
  }
}

// Handle download current tab
async function handleDownloadCurrentTab() {
  try {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tabs.length === 0) {
      alert('No active tab found');
      return;
    }

    const currentTab = tabs[0];
    const url = currentTab.url || '';

    // Validate it's a Freepik URL
    const assetPatterns = [
      /freepik\.com\/(free|premium)-(photo|vector|psd|ai|video)\//,
      /freepik\.com\/.*\/.*\.html/,
    ];

    if (!assetPatterns.some(pattern => pattern.test(url))) {
      alert('Please open a Freepik asset page first');
      return;
    }

    // Show loading state
    const button = document.getElementById('downloadCurrentBtn');
    if (button) {
      button.disabled = true;
      button.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="spinning">
          <circle cx="12" cy="12" r="10"></circle>
        </svg>
        Downloading...
      `;
    }

    // Send download request
    const response = await new Promise((resolve, reject) => {
      chrome.runtime.sendMessage({
        action: 'download',
        url: url,
        captchaToken: null
      }, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
        } else {
          resolve(response || { success: false, error: 'No response' });
        }
      });
    });

    // Reset button
    if (button) {
      button.disabled = false;
      button.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
          <polyline points="7 10 12 15 17 10"></polyline>
          <line x1="12" y1="15" x2="12" y2="3"></line>
        </svg>
        Download with PixyMod
      `;
    }

    // Handle response
    if (response && response.success) {
      // Show success briefly
      if (button) {
        button.innerHTML = `
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M20 6L9 17l-5-5"></path>
          </svg>
          Downloaded! ✓
        `;
        setTimeout(() => {
          button.innerHTML = `
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
              <polyline points="7 10 12 15 17 10"></polyline>
              <line x1="12" y1="15" x2="12" y2="3"></line>
            </svg>
            Download with PixyMod
          `;
        }, 2000);
      }
    } else {
      const errorMsg = response?.message || response?.error || 'Download failed';
      alert(errorMsg);
    }

  } catch (error) {
    console.error('Download error:', error);
    alert('Download failed: ' + error.message);

    // Reset button
    const button = document.getElementById('downloadCurrentBtn');
    if (button) {
      button.disabled = false;
      button.innerHTML = `
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
          <polyline points="7 10 12 15 17 10"></polyline>
          <line x1="12" y1="15" x2="12" y2="3"></line>
        </svg>
        Download with PixyMod
      `;
    }
  }
}

// Load user stats
async function loadUserStats() {
  try {
    // Get pixymod.com tabs
    const pixyTabs = await chrome.tabs.query({
      url: [
        'https://pixymod.com/*',
        'https://www.pixymod.com/*',
        'http://localhost:3000/*',
        'http://127.0.0.1:3000/*'
      ]
    });

    if (!pixyTabs.length) {
      
      document.getElementById('downloadsToday').textContent = '0';
      document.getElementById('totalDownloads').textContent = '0';
      return;
    }

    const targetTab = pixyTabs[0];
    const API_BASE_URL = targetTab.url.includes('localhost') || targetTab.url.includes('127.0.0.1')
      ? 'http://localhost:3000'
      : 'https://pixymod.com';

    // Fetch stats using page context (to use cookies)
    const result = await chrome.scripting.executeScript({
      target: { tabId: targetTab.id },
      world: 'MAIN',
      func: async (apiBaseUrl) => {
        try {
          // Get token if available
          let token = null;
          if (typeof window.__pixymod_getToken__ === 'function') {
            try {
              token = await window.__pixymod_getToken__();
            } catch (e) {
              
            }
          }

          const headers = { 'Content-Type': 'application/json' };
          if (token) headers['Authorization'] = `Bearer ${token}`;

          // Fetch daily downloads
          const dailyRes = await fetch(`${apiBaseUrl}/api/user/daily-downloads`, {
            headers,
            credentials: 'include'
          });
          const dailyData = await dailyRes.json();

          return {
            success: dailyData.success,
            downloadsToday: dailyData.success ? (dailyData.data?.dailyDownloads || 0) : 0,
            totalDownloads: dailyData.success ? (dailyData.data?.dailyDownloads || 0) : 0 // Use same for now
          };
        } catch (error) {
          console.error('Stats error:', error);
          return { success: false, downloadsToday: 0, totalDownloads: 0 };
        }
      },
      args: [API_BASE_URL]
    });

    const statsData = result?.[0]?.result;
    if (statsData && statsData.success) {
      document.getElementById('downloadsToday').textContent = statsData.downloadsToday.toString();
      document.getElementById('totalDownloads').textContent = statsData.totalDownloads.toString();
    } else {
      document.getElementById('downloadsToday').textContent = '0';
      document.getElementById('totalDownloads').textContent = '0';
    }
  } catch (error) {
    console.error('Error loading stats:', error);
    document.getElementById('downloadsToday').textContent = '0';
    document.getElementById('totalDownloads').textContent = '0';
  }
}

// Get download history
async function getDownloadHistory() {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'getDownloadHistory' }, (response) => {
      resolve(response?.history || []);
    });
  });
}

// Show history modal
function showHistoryModal(history) {
  // TODO: Create modal to show download history
  // For now, just log it
  
  alert(`You have ${history.length} downloads in history`);
}

