// Content script for pixymod.com - syncs auth state with extension

(function() {
  'use strict';

  

  // Listen for messages from extension
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'getAuthState') {
      getAuthState()
        .then(state => sendResponse({ success: true, ...state }))
        .catch(error => sendResponse({ success: false, error: error.message }));
      return true; // Keep channel open for async
    }
    
    if (request.action === 'getAuthToken') {
      
      getAuthToken()
        .then(token => {
          
          sendResponse({ success: true, token: token || null });
        })
        .catch(error => {
          console.error('[PixyMod Content] Error getting token:', error);
          sendResponse({ success: false, error: error.message, token: null });
        });
      return true; // Keep channel open for async response
    }
  });

  // Get Firebase auth state
  async function getAuthState() {
    // Wait a bit for Firebase to initialize
    await waitForFirebase();
    
    // Check if Firebase auth is available
    if (typeof window !== 'undefined' && window.firebase) {
      const auth = window.firebase.auth();
      const user = auth.currentUser;
      
      if (user) {
        return {
          isAuthenticated: true,
          user: {
            uid: user.uid,
            email: user.email,
            displayName: user.displayName,
            photoURL: user.photoURL,
            emailVerified: user.emailVerified
          }
        };
      }
    }
    
    // Fallback: check if user element exists in DOM
    const userElement = document.querySelector('[data-user]') || 
                        document.querySelector('.user-email') ||
                        document.querySelector('[class*="user"]');
    
    if (userElement) {
      // Try to extract user info from page
      const email = userElement.textContent || userElement.getAttribute('data-email');
      if (email && email.includes('@')) {
        return {
          isAuthenticated: true,
          user: { email: email }
        };
      }
    }
    
    return {
      isAuthenticated: false,
      user: null
    };
  }

  // Get Firebase ID token - access Firebase from React context or module
  async function getAuthToken() {
    try {
      
      
      
      
      
      
      // Wait longer for functions to be available (they're set before React and in useEffect)
      for (let i = 0; i < 100; i++) { // 10 seconds total
        // Method 1: Use the exposed function from Frontend (most reliable)
        if (typeof window !== 'undefined' && typeof window.__pixymod_getToken__ === 'function') {
          
          try {
            const token = await window.__pixymod_getToken__();
            
            if (token && typeof token === 'string' && token.split('.').length === 3) {
              
              return token;
            } else {
              
            }
          } catch (err) {
            console.error('[PixyMod Content] Error calling __pixymod_getToken__:', err);
          }
        }
        
        // Method 2: Access exposed Firebase auth instance
        if (typeof window !== 'undefined' && window.__pixymod_firebase_auth__) {
          
          try {
            const auth = window.__pixymod_firebase_auth__;
            
            // Wait for auth state if no user
            if (!auth.currentUser) {
              await new Promise((resolve) => {
                const unsubscribe = auth.onAuthStateChanged((user) => {
                  unsubscribe();
                  resolve();
                });
                setTimeout(() => {
                  unsubscribe();
                  resolve();
                }, 2000);
              });
            }
            
            const user = auth.currentUser;
            
            if (user) {
              
              const token = await user.getIdToken(true); // Force refresh
              if (token && typeof token === 'string' && token.split('.').length === 3) {
                
                return token;
              } else {
                
              }
            } else {
              
            }
          } catch (err) {
            console.error('[PixyMod Content] Error accessing Firebase auth:', err);
          }
        }
        
        // If functions aren't ready yet, wait a bit
        if (i < 99) {
          await new Promise(resolve => setTimeout(resolve, 100));
        }
      }

      
      
      
      return null;
    } catch (error) {
      console.error('[PixyMod Content] Error getting auth token:', error);
      return null;
    }
  }

  // Inject script to access Firebase auth from the page's context
  async function injectAndGetToken() {
    return new Promise((resolve) => {
      // Wait for exposed functions to be available
      let attempts = 0;
      const maxAttempts = 30; // 3 seconds
      
      const checkForToken = () => {
        attempts++;
        
        // Check if exposed functions are available
        if (typeof window.__pixymod_getToken__ === 'function') {
          window.__pixymod_getToken__().then(token => {
            resolve(token || null);
          }).catch(() => {
            if (attempts >= maxAttempts) {
              resolve(null);
            } else {
              setTimeout(checkForToken, 100);
            }
          });
          return;
        }
        
        if (window.__pixymod_firebase_auth__) {
          const auth = window.__pixymod_firebase_auth__;
          const user = auth.currentUser;
          if (user) {
            user.getIdToken().then(token => {
              resolve(token);
            }).catch(() => {
              if (attempts >= maxAttempts) {
                resolve(null);
              } else {
                setTimeout(checkForToken, 100);
              }
            });
            return;
          }
        }
        
        if (attempts >= maxAttempts) {
          resolve(null);
        } else {
          setTimeout(checkForToken, 100);
        }
      };
      
      checkForToken();
    });
  }

  // Wait for Firebase to be available
  function waitForFirebase() {
    return new Promise((resolve) => {
      if (typeof window !== 'undefined' && window.firebase && window.firebase.auth) {
        resolve();
        return;
      }
      
      // Wait up to 3 seconds for Firebase
      let attempts = 0;
      const checkInterval = setInterval(() => {
        attempts++;
        if (typeof window !== 'undefined' && window.firebase && window.firebase.auth) {
          clearInterval(checkInterval);
          resolve();
        } else if (attempts > 30) {
          clearInterval(checkInterval);
          resolve(); // Give up after 3 seconds
        }
      }, 100);
    });
  }

  // Monitor auth state changes
  if (typeof window !== 'undefined' && window.firebase) {
    window.firebase.auth().onAuthStateChanged((user) => {
      // Notify extension of auth state change
      chrome.runtime.sendMessage({
        action: 'authStateChanged',
        isAuthenticated: !!user,
        user: user ? {
          uid: user.uid,
          email: user.email,
          displayName: user.displayName
        } : null
      }).catch(() => {
        // Ignore errors if extension isn't listening
      });
    });
  }
})();

