// Content Script - Injected into Freepik.com pages
// Detects asset pages and adds download buttons

(function () {
  'use strict';

  

  // Configuration
  const CONFIG = {
    BUTTON_ID: 'pixymod-download-btn',
    BUTTON_CLASS: 'pixymod-download-button',
    CHECK_INTERVAL: 1000, // Check for new pages every second
  };

  // Initialize
  init();

  function init() {
    // Wait for page to load
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', addDownloadButton);
    } else {
      addDownloadButton();
    }

    // Watch for dynamic content changes (SPA navigation)
    observePageChanges();

    // Periodic check every 2 seconds as failsafe
    setInterval(() => {
      if (getCurrentAssetUrl() && !document.getElementById(CONFIG.BUTTON_ID)) {
        
        addDownloadButton();
      }
    }, 2000);
  }

  // Add download button to page
  function addDownloadButton(force = false) {
    // Check if button already exists (but allow force re-injection)
    const existingButton = document.getElementById(CONFIG.BUTTON_ID);
    if (existingButton && !force) {
      
      return;
    }

    // If forcing, remove existing button first
    if (existingButton && force) {
      
      existingButton.remove();
    }

    // Check if this is an asset page
    const assetUrl = getCurrentAssetUrl();
    
    
    if (!assetUrl) {
      return;
    }

    // Try to find injection point with retries (modals might load async)
    let retryCount = 0;
    const maxRetries = 20; // Increased retries
    const retryDelay = 300; // Faster retries

    const attemptInject = () => {
      const targetElement = findInjectionPoint();

      if (!targetElement || targetElement === document.body) {
        retryCount++;
        if (retryCount < maxRetries) {
          
          setTimeout(attemptInject, retryDelay);
          return;
        } else {
          console.warn('[PixyMod] Could not find injection point after retries');
          return;
        }
      }

      // Double-check button doesn't exist before creating
      if (document.getElementById(CONFIG.BUTTON_ID)) {
        
        return;
      }

      
      // Create download button
      const button = createDownloadButton(assetUrl);

      // Insert button
      insertButton(targetElement, button);
      
    };

    // Start injection attempts
    attemptInject();
  }

  // Get current Freepik asset URL
  function getCurrentAssetUrl() {
    const url = window.location.href;

    // Check if URL matches Freepik asset pattern
    // Examples:
    // - https://www.freepik.com/free-photo/...
    // - https://www.freepik.com/premium-vector/...
    // - https://www.freepik.com/premium-vector/flat-stickers-collection-earth-day-celebration_38898422.htm

    const assetPatterns = [
      /freepik\.com\/(free|premium)-(photo|vector|psd|ai|video)\//,
      /freepik\.com\/.*\/.*\.html?$/,  // .htm or .html
      /freepik\.com\/(free|premium)-(photo|vector|psd|ai|video)\/.*_\d+\.htm$/,  // _NUMBER.htm pattern
    ];

    const matches = assetPatterns.some(pattern => pattern.test(url));
    
    return matches ? url : null;
  }

  // Find best place to inject download button
  function findInjectionPoint() {
    // Priority 1: Find buttons in modal/popup (where preview shows)
    // Look for dialog/overlay containers first
    const modalSelectors = [
      '[role="dialog"]',
      '[class*="modal"]',
      '[class*="Modal"]',
      '[class*="dialog"]',
      '[class*="Dialog"]',
      '[class*="overlay"]',
      '[class*="Overlay"]',
      '[class*="popup"]',
      '[class*="Popup"]'
    ];

    for (const modalSelector of modalSelectors) {
      const modal = document.querySelector(modalSelector);
      if (modal) {
        // Look for download button or button container within the modal
        const modalDownloadBtn = modal.querySelector('[data-testid="download-button"]') ||
          modal.querySelector('button[aria-label*="Download" i]') ||
          modal.querySelector('[class*="download"]');
        if (modalDownloadBtn) {
          
          return modalDownloadBtn;
        }
      }
    }

    // Priority 2: Try standard page selectors
    const selectors = [
      '[data-testid="download-button"]',
      'button[aria-label*="Transparent PNG" i]',
      'button[aria-label*="Edit with AI" i]',
      'button[aria-label*="Edit with Assistant" i]',
      '.download-button',
      '.premium-download',
      '[class*="download"]',
      '[class*="Download"]',
      'button[aria-label*="download" i]',
      'button:has(svg)',  // Buttons with SVG icons
      '.actions',
      '.toolbar',
      '[class*="actions"]',
      '[class*="Actions"]',
      '[class*="button-group"]',
      '[class*="ButtonGroup"]',
      // Non-logged in / Premium selectors
      '[data-testid="go-premium"]',
      'a[href*="pricing"]',
      'a[href*="profile/register"]',
      'button[class*="premium"]',
      // 'aside' removed as it was selecting the left sidebar
      '[class*="detail-actions"]'
    ];

    for (const selector of selectors) {
      const element = document.querySelector(selector);
      if (element) {
        
        return element;
      }
    }

    // Fallback: Try to find main content area
    
    return document.querySelector('main') || document.body;
  }

  // Create download button element
  function createDownloadButton(assetUrl) {
    const button = document.createElement('button');
    button.id = CONFIG.BUTTON_ID;
    button.className = CONFIG.BUTTON_CLASS;
    button.innerHTML = `
      <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
        <polyline points="7 10 12 15 17 10"></polyline>
        <line x1="12" y1="15" x2="12" y2="3"></line>
      </svg>
      <span>Download with PixyMod</span>
    `;

    button.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      
      handleDownloadClick(assetUrl);
    });

    return button;
  }

  // Insert button into page
  function insertButton(targetElement, button) {
    // Strategy: Always insert PixyMod button FIRST in the button group
    // This ensures it appears before the native Download button

    // First, try to find the button container/parent
    let buttonContainer = null;

    // If targetElement is already a button, get its parent
    if (targetElement.matches?.('button, [role="button"], a[href*="download"]')) {
      buttonContainer = targetElement.parentElement;
    } else {
      // targetElement might be a container, use it directly
      buttonContainer = targetElement;
    }

    // Find all existing download-related buttons in the container
    const existingButtons = buttonContainer?.querySelectorAll('button, [role="button"], a[href*="download"]');

    if (existingButtons && existingButtons.length > 0) {
      // Insert PixyMod button BEFORE the first button found
      const firstButton = existingButtons[0];
      if (firstButton.parentElement) {
        firstButton.parentElement.insertBefore(button, firstButton);
        
        return;
      }
    }

    // Fallback: Try to find the native download button more specifically
    const nativeDownloadBtn =
      document.querySelector('[data-testid="download-button"]') ||
      document.querySelector('button[aria-label*="Download" i]') ||
      document.querySelector('a[href*="download"]');

    if (nativeDownloadBtn && nativeDownloadBtn.parentElement) {
      nativeDownloadBtn.parentElement.insertBefore(button, nativeDownloadBtn);
      
      return;
    }

    // Fallback: Try to insert at the beginning of the toolbar/actions container
    const toolbarContainer =
      targetElement.closest?.('.toolbar, .actions, [class*="toolbar"], [class*="Toolbar"], [class*="actions"], [class*="Actions"]') ||
      targetElement.parentElement ||
      null;

    if (toolbarContainer) {
      // Insert as the first child of the toolbar
      toolbarContainer.insertBefore(button, toolbarContainer.firstChild);
      
      return;
    }

    // Last resort: append to body
    console.warn('[PixyMod] Could not find proper insertion point, appending to body');
    document.body.appendChild(button);
  }

  // Handle download button click
  async function handleDownloadClick(url) {
    // 
    try {
      // Check if extension context is still valid
      if (!chrome.runtime?.id) {
        // console.error('[PixyMod] Extension context invalidated - please reload the page');
        alert('Extension was reloaded. Please refresh this page to continue.');
        return;
      }

      // Show loading state
      setButtonLoading(true);
      // 

      // Send download request directly - let API handle auth
      // Don't check auth first, let the API return error if needed
      const response = await new Promise((resolve, reject) => {
        try {
          // Check if runtime is still valid before sending
          if (!chrome.runtime || !chrome.runtime.id) {
            reject(new Error('Extension context invalidated. Please refresh this page.'));
            return;
          }

          chrome.runtime.sendMessage({
            action: 'download',
            url: url,
            captchaToken: null // Will be handled by API if needed
          }, (response) => {
            // 
            if (chrome.runtime.lastError) {
              const errorMsg = chrome.runtime.lastError.message;
              console.error('[PixyMod] Runtime error:', errorMsg);

              // Handle extension context invalidated error
              if (errorMsg.includes('Extension context invalidated') ||
                errorMsg.includes('message port closed') ||
                errorMsg.includes('Receiving end does not exist')) {
                reject(new Error('Extension was reloaded. Please refresh this page.'));
              } else {
                reject(new Error(errorMsg));
              }
            } else {
              resolve(response || { success: false, error: 'No response' });
            }
          });
        } catch (error) {
          console.error('[PixyMod] Error sending message:', error);
          reject(error);
        }
      });

      // Handle response exactly like Frontend DownloadForm does
      if (response && response.success) {
        // Success - show brief visual feedback
        showSuccessMessage();
        setTimeout(() => {
          setButtonLoading(false);
        }, 2000);
      } else {
        // Handle errors like Frontend does
        const errorMsg = response?.error || response?.message || '';
        const errorCode = response?.error;

        // Check for specific error types (same as Frontend)
        if (errorCode === 'Rate limit exceeded' || (errorMsg && errorMsg.toLowerCase().includes('please wait'))) {
          // Rate limit: show message, DO NOT redirect
          showErrorMessage(errorMsg || 'Rate limit exceeded. Please wait a few seconds and try again.');
        } else if (errorCode === 'Daily download limit exceeded') {
          // Daily limit: open pricing
          chrome.runtime.sendMessage({
            action: 'openTab',
            url: 'https://pixymod.com/pricing'
          }).catch(() => {
            window.open('https://pixymod.com/pricing', '_blank');
          });
        } else if (errorMsg.includes('sign in') || errorMsg.includes('login') || errorMsg.includes('Unauthorized') || errorCode === 'Unauthorized') {
          // Auth error - silently open login page
          chrome.runtime.sendMessage({
            action: 'openTab',
            url: 'https://pixymod.com/freepik-downloader'
          }).catch(() => {
            window.open('https://pixymod.com/freepik-downloader', '_blank');
          });
        }
        setButtonLoading(false);
        console.error('Download failed:', errorMsg);
      }
    } catch (error) {
      console.error('[PixyMod] Download error:', error);
      setButtonLoading(false);

      // Handle extension context invalidated error
      if (error.message && (
        error.message.includes('Extension context invalidated') ||
        error.message.includes('Extension was reloaded') ||
        error.message.includes('message port closed')
      )) {
        // Show user-friendly message
        const toast = document.createElement('div');
        toast.className = 'pixymod-toast error';
        toast.innerHTML = '<strong>Extension Reloaded</strong><br>Please refresh this page to continue downloading.';
        document.body.appendChild(toast);
        setTimeout(() => toast.remove(), 5000);
      }
    }
  }

  // Check authentication status
  async function checkAuthStatus() {
    return new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'checkAuth' }, (response) => {
        if (chrome.runtime.lastError) {
          console.error('Error checking auth:', chrome.runtime.lastError);
          resolve({ isAuthenticated: false });
        } else {
          resolve(response || { isAuthenticated: false });
        }
      });
    });
  }

  // Get reCAPTCHA token (if needed)
  async function getCaptchaToken() {
    // TODO: Implement reCAPTCHA if needed
    // For now, return null (premium users skip this)
    return null;
  }

  // Set button loading state
  function setButtonLoading(loading) {
    const button = document.getElementById(CONFIG.BUTTON_ID);
    if (!button) return;

    if (loading) {
      button.disabled = true;
      button.classList.add('loading');
      const span = button.querySelector('span');
      if (span) span.textContent = 'Downloading...';
    } else {
      button.disabled = false;
      button.classList.remove('loading');
      button.querySelector('span').textContent = 'Download with PixyMod';
    }
  }

  // Show success message (brief visual feedback only)
  function showSuccessMessage() {
    const button = document.getElementById(CONFIG.BUTTON_ID);
    if (button) {
      button.classList.add('success');
      const span = button.querySelector('span');
      if (span) {
        span.textContent = 'Downloaded! ✓';
      }

      setTimeout(() => {
        button.classList.remove('success');
        if (span) {
          span.textContent = 'Download with PixyMod';
        }
      }, 2000);
    }
  }

  // Show error message (removed - no alerts needed)
  function showErrorMessage(message) {
    console.error('Download error:', message);
    const button = document.getElementById(CONFIG.BUTTON_ID);
    if (button) {
      const span = button.querySelector('span');
      if (span) span.textContent = message;
      setTimeout(() => {
        if (span) span.textContent = 'Download with PixyMod';
      }, 2500);
    }
  }

  // Show login prompt (removed - silent redirect)
  function showLoginPrompt() {
    // Silently open login page in background
    chrome.runtime.sendMessage({
      action: 'openTab',
      url: 'https://pixymod.com/freepik-downloader'
    }).catch(error => {
      console.error('Error opening tab:', error);
      // Fallback: open in same window
      window.open('https://pixymod.com/freepik-downloader', '_blank');
    });
  }

  // Observe page changes (for SPA navigation)
  function observePageChanges() {
    // Use MutationObserver to detect DOM changes
    const observer = new MutationObserver((mutations) => {
      // Check if any mutations added a modal/dialog
      let modalAdded = false;
      for (const mutation of mutations) {
        if (mutation.addedNodes.length > 0) {
          for (const node of mutation.addedNodes) {
            if (node.nodeType === Node.ELEMENT_NODE) {
              const el = node;
              // Check if it's a modal/dialog or contains one
              if (el.matches?.('[role="dialog"], [class*="modal"], [class*="Modal"], [class*="overlay"], [class*="Overlay"]') ||
                el.querySelector?.('[role="dialog"], [class*="modal"], [class*="Modal"]')) {
                modalAdded = true;
                
                break;
              }
            }
          }
        }
        if (modalAdded) break;
      }

      // If modal was added, inject immediately, otherwise debounce
      const delay = modalAdded ? 100 : 300;

      setTimeout(() => {
        if (!document.getElementById(CONFIG.BUTTON_ID)) {
          addDownloadButton();
        }
      }, delay);
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });

    // Also listen for URL changes (SPA navigation)
    let lastUrl = location.href;
    setInterval(() => {
      const currentUrl = location.href;
      if (currentUrl !== lastUrl) {
        lastUrl = currentUrl;
        // Remove old button if exists
        const oldButton = document.getElementById(CONFIG.BUTTON_ID);
        if (oldButton) {
          oldButton.remove();
        }
        // Add new button
        setTimeout(addDownloadButton, 1000);
      }
    }, CONFIG.CHECK_INTERVAL);
  }

})();

