const API_BASE_URL = 'https://pixymod.com'; // Update with your domain
// Initialize Firebase (you'll need to include Firebase SDK)
// import { initializeApp } from 'firebase/app';
// import { getAuth } from 'firebase/auth';

// Storage keys
const STORAGE_KEYS = {
  AUTH_TOKEN: 'auth_token',
  USER_INFO: 'user_info',
  DOWNLOAD_HISTORY: 'download_history',
  SETTINGS: 'settings'
};



// Listen for messages from content script or popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  
  
  if (request.action === 'download') {
    
    
    // Handle download with async/await
    (async () => {
      try {
        const result = await handleDownload(request.url, request.captchaToken);
        
        // Result already has success/error structure from handleDownload
        sendResponse(result);
      } catch (error) {
        console.error('[PixyMod Background] Download error:', error);
        sendResponse({ 
          success: false, 
          error: error.message || 'Download failed',
          message: error.message || 'An unexpected error occurred'
        });
      }
    })();
    
    return true; // Keep channel open for async response
  }

  if (request.action === 'checkAuth') {
    // Handle async response properly
    (async () => {
      try {
        const status = await checkAuthStatus();
        sendResponse({ success: true, ...status });
      } catch (error) {
        console.error('[PixyMod Background] checkAuth error:', error);
        sendResponse({ success: false, error: error.message, isAuthenticated: false });
      }
    })();
    return true; // Keep channel open for async response
  }

  if (request.action === 'getDownloadHistory') {
    getDownloadHistory()
      .then(history => sendResponse({ success: true, history }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }

  if (request.action === 'authStateChanged') {
    // Auth state changed on pixymod.com, update storage
    if (request.isAuthenticated && request.user) {
      chrome.storage.local.set({
        [STORAGE_KEYS.USER_INFO]: request.user
      });
    } else {
      // User logged out, clear storage
      chrome.storage.local.remove([STORAGE_KEYS.AUTH_TOKEN, STORAGE_KEYS.USER_INFO]);
    }
    return false;
  }

  if (request.action === 'storeAuth') {
    // Store auth info from popup
    if (request.user) {
      chrome.storage.local.set({
        [STORAGE_KEYS.USER_INFO]: request.user
      });
      sendResponse({ success: true });
    } else {
      sendResponse({ success: false });
    }
    return true;
  }

  if (request.action === 'openTab') {
    // Open a new tab (called from content script)
    chrome.tabs.create({ url: request.url })
      .then(() => sendResponse({ success: true }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

// Handle download request - matches Frontend implementation
async function handleDownload(url, captchaToken = null) {
  
  
  
  
  
  
  try {
    // Step 2: subscription + request body (same as Frontend)
    
    const userInfo = await getUserInfo();
    const hasActiveSubscription = userInfo?.hasActiveSubscription || false;
    
    

    
    const requestBody = {
      url,
      captchaToken: hasActiveSubscription ? null : captchaToken
    };

    // Step 4: Always call API from pixymod.com tab context (cookies auto-send).
    // If cookie auth fails (401), retry using window.__pixymod_getToken__() from the same tab.
    

    const pixyTabs = await chrome.tabs.query({
      url: [
        'https://pixymod.com/*',
        'https://www.pixymod.com/*',
        'http://localhost:3000/*',
        'http://127.0.0.1:3000/*'
      ]
    });

    if (!pixyTabs.length) {
      return {
        success: false,
        error: 'Please login to download',
        message: 'Please open pixymod.com in a tab and log in, then try again.'
      };
    }

    const targetTab =
      pixyTabs.find(t => (t.url || '').includes('/freepik-downloader')) ||
      pixyTabs[0];

    if (!targetTab?.id) {
      return {
        success: false,
        error: 'Authentication failed',
        message: 'Could not access pixymod.com tab. Please reload pixymod.com and try again.'
      };
    }

    const apiUrl = `${API_BASE_URL}/api/download`;
    const exec = await chrome.scripting.executeScript({
      target: { tabId: targetTab.id },
      // IMPORTANT: run in MAIN world so we can access window.__pixymod_getToken__()
      world: 'MAIN',
      func: async (apiUrl, requestBodyJson) => {
        const body = JSON.parse(requestBodyJson);

        const doFetch = async (extraHeaders = {}, allowRetry = true) => {
          const res = await fetch(apiUrl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', ...extraHeaders },
            credentials: 'include',
            body: JSON.stringify(body)
          });
          let data = null;
          try {
            data = await res.json();
          } catch (e) {
            data = { success: false, error: 'Invalid response', message: String(e) };
          }

          // Auto-handle backend rate limit (1 request / 4s): wait then retry once.
          if (allowRetry && res.status === 429 && data?.message) {
            const m = String(data.message).match(/Please wait\s+(\d+)\s+seconds?/i);
            const waitSeconds = m ? Number(m[1]) : 4;
            const waitMs = Math.max(1, waitSeconds) * 1000 + 200;
            await new Promise(r => setTimeout(r, waitMs));
            return await doFetch(extraHeaders, false);
          }

          return { status: res.status, data };
        };

        // 1) Try cookie auth first (fastest)
        let { status, data } = await doFetch();

        // 2) If unauthorized, try token from page (wait briefly for the function/user to be ready)
        const msg = (data?.message || '').toLowerCase();
        if (
          status === 401 ||
          data?.error === 'Unauthorized' ||
          msg.includes('please login') ||
          msg.includes('login')
        ) {
          let token = null;
          try {
            // Wait up to 3s for the function to exist (React hydration / useEffect timing)
            for (let i = 0; i < 30; i++) {
              if (typeof window.__pixymod_getToken__ === 'function') break;
              await new Promise(r => setTimeout(r, 100));
            }

            if (typeof window.__pixymod_getToken__ === 'function') {
              token = await window.__pixymod_getToken__();
            }
          } catch (e) {
            // ignore
          }

          if (token && typeof token === 'string') {
            ({ status, data } = await doFetch({ Authorization: `Bearer ${token}` }));
            return { used: 'token', status, data, tokenTried: true, tokenOk: true };
          }

          return { used: 'cookie', status, data, tokenTried: true, tokenOk: false, tokenFn: typeof window.__pixymod_getToken__ };
        }

        return { used: 'cookie', status, data };
      },
      args: [apiUrl, JSON.stringify(requestBody)]
    });

    const apiResult = exec?.[0]?.result;
    if (!apiResult) {
      return {
        success: false,
        error: 'API call failed',
        message: 'No response from pixymod.com tab API call.'
      };
    }

    

    const data = apiResult.data;
    const status = apiResult.status;

    if (!data?.success) {
      if (status === 401) {
        return {
          success: false,
          error: 'Please login to download',
          message: 'You appear logged in, but the server did not accept auth. Please refresh the pixymod.com tab and try again.'
        };
      }

      return {
        success: false,
        error: data?.error || 'Download failed',
        message: data?.message || data?.error || 'Download failed',
        currentDownloads: data?.currentDownloads,
        limit: data?.limit
      };
    }

    
    const downloadUrl = data.downloadUrl || data.fileUrl;
    const fileName = data.fileName || 'freepik-asset';

    if (!downloadUrl) {
      return {
        success: false,
        error: 'No download URL in response',
        message: 'Download URL not found in response'
      };
    }

    
    await downloadFile(downloadUrl, fileName);

    // Save to history (background operation, don't block)
    
    saveToHistory({
      url: url,
      fileName: fileName,
      downloadedAt: new Date().toISOString(),
      success: true
    }).catch(err => console.error('[PixyMod Background] ❌ Failed to save history:', err));

    // Update badge briefly
    
    updateBadge('success');
    clearBadge();

    
    
    
    
    

    // Return success with data (same structure as Frontend)
    return { 
      success: true, 
      downloadUrl: downloadUrl,
      fileName: fileName,
      ...data 
    };
  } catch (error) {
    console.error('═══════════════════════════════════════════════════════════');
    console.error('[PixyMod Background] ❌❌❌ DOWNLOAD FAILED ❌❌❌');
    console.error('[PixyMod Background] Error:', error.message);
    console.error('[PixyMod Background] Error Stack:', error.stack);
    console.error('[PixyMod Background] Error Name:', error.name);
    console.error('═══════════════════════════════════════════════════════════');
    updateBadge('error');
    clearBadge();
    return {
      success: false,
      error: error.message || 'Download failed',
      message: error.message || 'An unexpected error occurred'
    };
  }
}

// Download file using Chrome Downloads API
async function downloadFile(url, filename) {
  return new Promise((resolve, reject) => {
    // Ensure filename has proper extension
    let finalFilename = filename;
    if (!finalFilename.includes('.')) {
      // Try to get extension from URL
      const urlMatch = url.match(/\.([a-zA-Z0-9]+)(\?|$)/);
      if (urlMatch) {
        finalFilename = `${filename}.${urlMatch[1]}`;
      } else {
        finalFilename = `${filename}.zip`; // Default to zip
      }
    }
    
    chrome.downloads.download({
      url: url,
      filename: finalFilename,
      saveAs: false, // Silent download, no prompt
      conflictAction: 'uniquify' // Auto-rename if file exists
    }, (downloadId) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve(downloadId);
      }
    });
  });
}

// Check authentication status
async function checkAuthStatus() {
  try {
    // Check for session cookie from pixymod.com FIRST (most reliable)
    let sessionCookie = null;
    try {
      const cookies = await chrome.cookies.getAll({ domain: 'pixymod.com' });
      sessionCookie = cookies.find(c => c.name === 'session');
      
    } catch (cookieError) {
      console.error('Error checking cookies:', cookieError);
    }
    
    // If we have a session cookie, user is definitely logged in
    if (sessionCookie && sessionCookie.value) {
      
      
      // Try to get user info from pixymod.com page (with timeout)
      let userInfoFromPage = null;
      try {
        const pageInfoPromise = getUserInfoFromPage();
        const timeoutPromise = new Promise((_, reject) => 
          setTimeout(() => reject(new Error('Page info timeout')), 5000)
        );
        userInfoFromPage = await Promise.race([pageInfoPromise, timeoutPromise]);
      } catch (pageError) {
        
      }
      
      // Also try to get token (with timeout to prevent hanging)
      let token = null;
      try {
        const tokenPromise = getAuthToken();
        const timeoutPromise = new Promise((_, reject) => 
          setTimeout(() => reject(new Error('Token timeout')), 8000)
        );
        token = await Promise.race([tokenPromise, timeoutPromise]);
        if (token && typeof token === 'string' && token.split('.').length === 3) {
          // Store token if valid
          await chrome.storage.local.set({ [STORAGE_KEYS.AUTH_TOKEN]: token });
        } else {
          token = null; // Ensure it's null if invalid
        }
      } catch (tokenError) {
        
        token = null;
      }
      
      // Store user info if we got it
      if (userInfoFromPage) {
        await chrome.storage.local.set({ [STORAGE_KEYS.USER_INFO]: userInfoFromPage });
      }
      
      return {
        isAuthenticated: true,
        user: userInfoFromPage || { email: 'User', displayName: 'User' },
        hasActiveSubscription: userInfoFromPage?.hasActiveSubscription || false
      };
    }
    
    // Fallback: check stored token (with timeout to prevent hanging)
    try {
      const tokenPromise = getAuthToken();
      const timeoutPromise = new Promise((_, reject) => 
        setTimeout(() => reject(new Error('Token retrieval timeout')), 5000)
      );
      
      const token = await Promise.race([tokenPromise, timeoutPromise]);
      const userInfo = await getUserInfo();
      
      if (token && userInfo) {
        return {
          isAuthenticated: true,
          user: userInfo,
          hasActiveSubscription: userInfo?.hasActiveSubscription || false
        };
      }
    } catch (tokenError) {
      
    }
    
    // Last fallback: try to get auth from pixymod.com page directly (with timeout)
    try {
      const pageInfoPromise = getUserInfoFromPage();
      const timeoutPromise = new Promise((_, reject) => 
        setTimeout(() => reject(new Error('Page info timeout')), 5000)
      );
      
      const userInfoFromPage = await Promise.race([pageInfoPromise, timeoutPromise]);
      if (userInfoFromPage) {
        return {
          isAuthenticated: true,
          user: userInfoFromPage,
          hasActiveSubscription: userInfoFromPage?.hasActiveSubscription || false
        };
      }
    } catch (pageError) {
      
    }
    
    
    return {
      isAuthenticated: false,
      user: null,
      hasActiveSubscription: false
    };
  } catch (error) {
    console.error('Auth check error:', error);
    return {
      isAuthenticated: false,
      user: null,
      hasActiveSubscription: false
    };
  }
}

// Get user info by messaging pixymod.com content script
async function getUserInfoFromPage() {
  try {
    // Find any pixymod.com tab
    const tabs = await chrome.tabs.query({ url: 'https://pixymod.com/*' });
    
    if (tabs.length > 0) {
      try {
        // Send message to content script with timeout
        const response = await Promise.race([
          chrome.tabs.sendMessage(tabs[0].id, { action: 'getAuthState' }),
          new Promise((_, reject) => 
            setTimeout(() => reject(new Error('Timeout')), 2000)
          )
        ]);
        
        if (response && response.success && response.isAuthenticated) {
          return response.user;
        }
      } catch (messageError) {
        // Content script might not be loaded, that's okay
        
      }
    }
    
    return null;
  } catch (error) {
    console.error('Error getting user info from page:', error);
    return null;
  }
}

// Get auth token from storage or from pixymod.com page
async function getAuthToken() {
  
  
  // First check storage
  const result = await chrome.storage.local.get([STORAGE_KEYS.AUTH_TOKEN]);
  if (result[STORAGE_KEYS.AUTH_TOKEN]) {
    
    // Verify token is still valid by checking if it's a valid JWT format
    const token = result[STORAGE_KEYS.AUTH_TOKEN];
    if (token && typeof token === 'string' && token.split('.').length === 3) {
      
      return token;
    } else {
      
      await chrome.storage.local.remove([STORAGE_KEYS.AUTH_TOKEN]);
    }
  }
  
  
  
  // Check if we have a session cookie - if yes, we can use it, but we still need ID token for API
  try {
    const cookies = await chrome.cookies.getAll({ domain: 'pixymod.com' });
    const sessionCookie = cookies.find(c => c.name === 'session');
    
  } catch (cookieError) {
    
  }
  
  // Try to get token from pixymod.com page via injected script
  try {
    // Query for all pixymod.com tabs (including www subdomain and localhost for dev)
    const tabs = await chrome.tabs.query({ 
      url: ['https://pixymod.com/*', 'https://www.pixymod.com/*', 'http://localhost:3000/*', 'http://127.0.0.1:3000/*'] 
    });
    
    
    if (tabs.length === 0) {
      
      return null;
    }
    
    if (tabs.length > 0) {
      try {
        // First try: Use content script message (faster, already loaded)
        // But first, ensure content script is loaded by injecting it if needed
        try {
          
          
          
          // Try to inject content script if it's not loaded
          try {
            await chrome.scripting.executeScript({
              target: { tabId: tabs[0].id },
              files: ['pixymod-content.js']
            });
            
            // Wait a bit for script to initialize
            await new Promise(resolve => setTimeout(resolve, 1000));
          } catch (injectError) {
            
            // Clear runtime error if it's just "already injected"
            if (chrome.runtime.lastError) {
              const errorMsg = chrome.runtime.lastError.message;
              if (errorMsg.includes('Cannot access') || errorMsg.includes('already')) {
                
              }
            }
          }
          
          const response = await Promise.race([
            chrome.tabs.sendMessage(tabs[0].id, { action: 'getAuthToken' }),
            new Promise((_, reject) => 
              setTimeout(() => reject(new Error('Content script timeout after 10s')), 10000)
            )
          ]);
          
          
          
          if (response && response.success && response.token) {
            const token = response.token;
            
            if (token && typeof token === 'string' && token.split('.').length === 3) {
              
              await chrome.storage.local.set({ [STORAGE_KEYS.AUTH_TOKEN]: token });
              return token;
            } else {
              
            }
          } else {
            
          }
        } catch (messageError) {
          
          // Check if it's a runtime error
          if (chrome.runtime.lastError) {
            
          }
          // Don't give up - try injected script next
        }
        
        // Second try: Inject script to get Firebase ID token directly from the page
        // This runs in page context, so it can access the page's Firebase instance
        
        
        
        // First, ensure the page is fully loaded
        try {
          await chrome.scripting.executeScript({
            target: { tabId: tabs[0].id },
            func: () => {
              // Wait for page to be ready
              return new Promise((resolve) => {
                if (document.readyState === 'complete') {
                  resolve();
                } else {
                  window.addEventListener('load', () => resolve(), { once: true });
                  setTimeout(() => resolve(), 2000); // Max 2 seconds wait
                }
              });
            }
          });
          
        } catch (e) {
          
        }
        
        try {
          const results = await Promise.race([
            chrome.scripting.executeScript({
              target: { tabId: tabs[0].id },
              func: async () => {
                try {
                  
                  
                  
                  
                  
                  
                  
                  // Method 0: Try to access Firebase directly from window (if available)
                  
                  if (window.firebase && window.firebase.auth) {
                    
                    try {
                      const auth = window.firebase.auth();
                      const user = auth.currentUser;
                      if (user) {
                        
                        const token = await user.getIdToken(true);
                        if (token && typeof token === 'string' && token.split('.').length === 3) {
                          
                          return token;
                        }
                      }
                    } catch (e) {
                      
                    }
                  }
                  
                  // Method 0.5: Try to access React context directly (if available)
                  if (window.__REACT_DEVTOOLS_GLOBAL_HOOK__) {
                    
                    try {
                      // Try to find React root and access AuthContext
                      const reactRoots = document.querySelectorAll('[data-reactroot], #__next, #root');
                      
                    } catch (e) {
                      
                    }
                  }
                  
                  // Wait longer for functions to be available (they're set in useEffect)
                  // Check immediately first, then wait if needed
                  
                  
                  // First, wait for React to hydrate (check for Next.js hydration)
                  let reactReady = false;
                  for (let j = 0; j < 50; j++) {
                    // Check if React has hydrated by looking for Next.js markers
                    if (document.querySelector('#__next') || window.__NEXT_DATA__) {
                      // Wait a bit more for useEffect to run
                      await new Promise(resolve => setTimeout(resolve, 500));
                      reactReady = true;
                      
                      break;
                    }
                    await new Promise(resolve => setTimeout(resolve, 100));
                  }
                  
                  if (!reactReady) {
                    
                  }
                  
                  // Now wait for exposed functions (up to 15 seconds total)
                  for (let i = 0; i < 150; i++) { // Increased to 150 attempts (15 seconds)
                    // Method 1: Use the exposed function from Frontend (most reliable)
                    if (typeof window.__pixymod_getToken__ === 'function') {
                      
                      try {
                        const token = await window.__pixymod_getToken__();
                        
                        if (token && typeof token === 'string' && token.split('.').length === 3) {
                          
                          return token;
                        } else {
                          
                        }
                      } catch (err) {
                        console.error('[PixyMod Page] Error calling __pixymod_getToken__:', err);
                      }
                    }
                    
                    // Method 2: Access exposed Firebase auth instance
                    if (window.__pixymod_firebase_auth__) {
                      
                      try {
                        const auth = window.__pixymod_firebase_auth__;
                        
                        // Wait for auth to be ready
                        if (!auth.currentUser) {
                          await new Promise((resolve) => {
                            const unsubscribe = auth.onAuthStateChanged((user) => {
                              unsubscribe();
                              resolve();
                            });
                            setTimeout(() => {
                              unsubscribe();
                              resolve();
                            }, 1000);
                          });
                        }
                        
                        const user = auth.currentUser;
                        
                        if (user) {
                          
                          const token = await user.getIdToken(true); // Force refresh
                          
                          if (token && typeof token === 'string' && token.split('.').length === 3) {
                            
                            return token;
                          } else {
                            
                          }
                        } else {
                          
                        }
                      } catch (err) {
                        console.error('[PixyMod Page] Error accessing Firebase auth:', err);
                      }
                    }
                    
                    // If functions aren't ready yet, wait a bit
                    if (i < 99) {
                      await new Promise(resolve => setTimeout(resolve, 100));
                    }
                  }
                  
                  
                  
                  
                  
                  
                  // Last resort: Try to find Firebase in any way possible
                  
                  
                  
                  
                  // Check if there's a way to access Firebase through Next.js
                  if (window.__NEXT_DATA__) {
                    
                  }
                  
                  return null;
                } catch (error) {
                  console.error('[PixyMod Page] Error getting token:', error);
                  return null;
                }
              }
            }),
            new Promise((_, reject) => 
              setTimeout(() => reject(new Error('Script execution timeout after 20s')), 20000)
            )
          ]);
          
          if (results && results[0] && results[0].result) {
            const token = results[0].result;
            if (token && typeof token === 'string' && token.split('.').length === 3) {
              
              
              try {
                // Store it for future use (tokens expire, but we'll refresh when needed)
                await chrome.storage.local.set({ [STORAGE_KEYS.AUTH_TOKEN]: token });
                
              } catch (storageError) {
                console.error('[PixyMod Background] ❌ Failed to store token:', storageError);
              }
              return token;
            } else {
              
            }
          } else {
            
          }
        } catch (scriptError) {
          
          if (scriptError.message && scriptError.message.includes('timeout')) {
            
          }
        }
      } catch (innerError) {
        console.error('[PixyMod Background] Error in inner try block:', innerError);
      }
    }
  } catch (error) {
    console.error('[PixyMod Background] Error getting token from page:', error);
  }
  
  
  return null;
}

// Get user info from storage
async function getUserInfo() {
  const result = await chrome.storage.local.get([STORAGE_KEYS.USER_INFO]);
  return result[STORAGE_KEYS.USER_INFO];
}

// Save to download history
async function saveToHistory(item) {
  const result = await chrome.storage.local.get([STORAGE_KEYS.DOWNLOAD_HISTORY]);
  const history = result[STORAGE_KEYS.DOWNLOAD_HISTORY] || [];
  history.unshift(item);
  
  // Keep only last 100 items
  if (history.length > 100) {
    history.pop();
  }
  
  await chrome.storage.local.set({ [STORAGE_KEYS.DOWNLOAD_HISTORY]: history });
}

// Get download history
async function getDownloadHistory() {
  const result = await chrome.storage.local.get([STORAGE_KEYS.DOWNLOAD_HISTORY]);
  return result[STORAGE_KEYS.DOWNLOAD_HISTORY] || [];
}

// Validate Freepik URL
function isFreepikUrl(url) {
  try {
    const urlObj = new URL(url);
    return urlObj.hostname.includes('freepik.com');
  } catch {
    return false;
  }
}

// Update extension badge
function updateBadge(status) {
  const badges = {
    success: { text: '✓', color: '#10b981' },
    error: { text: '!', color: '#ef4444' },
    loading: { text: '...', color: '#6366f1' }
  };
  
  const badge = badges[status] || badges.loading;
  chrome.action.setBadgeText({ text: badge.text });
  chrome.action.setBadgeBackgroundColor({ color: badge.color });
}

// Clear badge after 3 seconds
function clearBadge() {
  setTimeout(() => {
    chrome.action.setBadgeText({ text: '' });
  }, 3000);
}

// Listen for tab updates to detect Freepik pages
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    if (isFreepikUrl(tab.url)) {
      // Update badge to show extension is active
      chrome.action.setBadgeText({ text: '', tabId: tabId });
    }
  }
});

// Install/Update handler
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    // First time install
    chrome.tabs.create({ url: `${API_BASE_URL}/freepik-downloader?extension=installed` });
  } else if (details.reason === 'update') {
    // Extension updated
    
  }
});

